package com.ylzinfo.onepay.sdk;

import java.util.List;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ylzinfo.onepay.sdk.domain.CancelParams;
import com.ylzinfo.onepay.sdk.domain.CancelResult;
import com.ylzinfo.onepay.sdk.domain.ChannelVO;
import com.ylzinfo.onepay.sdk.domain.ChargeParams;
import com.ylzinfo.onepay.sdk.domain.ChargeResult;
import com.ylzinfo.onepay.sdk.domain.ChargeVO;
import com.ylzinfo.onepay.sdk.domain.CloseParams;
import com.ylzinfo.onepay.sdk.domain.CloseResult;
import com.ylzinfo.onepay.sdk.domain.DownloadBillParams;
import com.ylzinfo.onepay.sdk.domain.DownloadBillResult;
import com.ylzinfo.onepay.sdk.domain.PosSignInfoParams;
import com.ylzinfo.onepay.sdk.domain.PosSignInfoResult;
import com.ylzinfo.onepay.sdk.domain.QRCodeParams;
import com.ylzinfo.onepay.sdk.domain.QRCodeResult;
import com.ylzinfo.onepay.sdk.domain.QueryParams;
import com.ylzinfo.onepay.sdk.domain.RefundParams;
import com.ylzinfo.onepay.sdk.domain.RefundResult;
import com.ylzinfo.onepay.sdk.domain.RefundVO;
import com.ylzinfo.onepay.sdk.domain.ResponseParams;
import com.ylzinfo.onepay.sdk.domain.SignInfoParams;
import com.ylzinfo.onepay.sdk.domain.SignInfoResult;
import com.ylzinfo.onepay.sdk.domain.SocialCardParams;
import com.ylzinfo.onepay.sdk.domain.TransferParams;
import com.ylzinfo.onepay.sdk.domain.TransferResult;
import com.ylzinfo.onepay.sdk.domain.WebHook;
import com.ylzinfo.onepay.sdk.domain.WristbandRefundParams;
import com.ylzinfo.onepay.sdk.domain.WristbandRefundResult;
import com.ylzinfo.onepay.sdk.enums.EncType;
import com.ylzinfo.onepay.sdk.enums.SignType;
import com.ylzinfo.onepay.sdk.exception.PayException;
import com.ylzinfo.onepay.sdk.service.OnepayService;
import com.ylzinfo.onepay.sdk.utils.SecurityUtil;
import com.ylzinfo.onepay.sdk.utils.StringUtil;

/**
 * 支付平台客户端
 * 
 * @author chenjh
 * 
 */
public class OnepayClient {
	/**
	 * 应用id
	 */
	private String appId;

	/**
	 * 应用安全key
	 */
	private String appSecret;

	/**
	 * 签名类型
	 */
	private String signType;

	/**
	 * 加密类型
	 */
	private String encryptType;

	/**
	 * 应用服务
	 */
	private OnepayService mainService;

	public OnepayClient(String onepayUrl, String appId, String appSecret) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = SignType.MD5.toString();
		this.encryptType = EncType.DES.toString();
		mainService = new OnepayService(onepayUrl, appId, appSecret, signType, encryptType);
	}

	public OnepayClient(String onepayUrl, String appId, String appSecret, String signType, String encryptType) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
		mainService = new OnepayService(onepayUrl, appId, appSecret, signType, encryptType);
	}

	public OnepayClient(String appId, String appSecret, String signType, String encryptType) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
		mainService = new OnepayService(appId, appSecret, signType, encryptType);
	}

	public String getAppId() {
		return appId;
	}

	public String getAppSecret() {
		return appSecret;
	}

	public String getSignType() {
		return signType;
	}

	public String getEncryptType() {
		return encryptType;
	}

	/**
	 * 订单支付
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime :
	 * @param payParams 支付请求参数
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<ChargeResult> charge(ChargeParams chargeParams) throws Exception {
		return mainService.charge(chargeParams);
	}

	/**
	 * 订单退款
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param refundParams 退款参数
	 * @return 退款返回
	 * @throws Exception
	 */
	public ResponseParams<RefundResult> refund(RefundParams refundParams) throws Exception {
		return mainService.refund(refundParams);
	}

	/**
	 * 订单关闭
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param closeParams 关闭参数
	 * @return 关闭返回
	 * @throws Exception
	 */
	public ResponseParams<CloseResult> close(CloseParams closeParams) throws Exception {
		return mainService.close(closeParams);
	}

	/**
	 * 撤销订单
	 */
	public ResponseParams<CancelResult> cancel(CancelParams cancelParams) throws Exception {
		return mainService.cancel(cancelParams);
	}

	/**
	 * 
	 * 查询订单列表
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param queryParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<List<ChargeVO>> queryChargeList(QueryParams queryParams) throws Exception {
		return mainService.queryChargeList(queryParams);
	}

	/**
	 * 查询退款列表
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param queryParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<List<RefundVO>> queryRefundList(QueryParams queryParams) throws Exception {
		return mainService.queryRefundList(queryParams);
	}

	/**
	 * 根据订单号查询订单
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param chargeNo  订单号
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<ChargeVO> queryCharge(String outChargeNo) throws Exception {
		return mainService.queryCharge(outChargeNo);
	}

	/**
	 * 根据退款流水号查询退款信息
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param refundNo 退款流水号
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<RefundVO> queryRefund(String outRefundNo) throws Exception {
		return mainService.queryRefund(outRefundNo);
	}

	/**
	 * 返回JSON字符串 验签
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime:
	 * @param responseMsg
	 * @return
	 * @throws Exception
	 * @Description:
	 */
	public boolean verifyResponseSign(String responseMsg) throws PayException {
		return mainService.verifySign(responseMsg);
	}

	/**
	 * 响应验签
	 * 
	 * @Description:
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime:
	 * @param responseParams
	 * @param appId
	 * @param appSecret
	 * @return
	 * @throws PayException
	 */
	public boolean verifyResponseSign(ResponseParams<?> responseParams) throws PayException {
		return mainService.verifySign(responseParams);
	}

	/**
	 * 解密异步通知响应消息
	 * 
	 * @Description:
	 * @Author: chenjh @Date: 2016年12月21日
	 * @LastEditTime:
	 * @param responseParams
	 * @param appId
	 * @param appSecret
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<WebHook> decryptWebHookResponse(ResponseParams<?> res) throws PayException {
		// 解密报文
		ResponseParams<WebHook> responseParams = new ResponseParams<WebHook>();

		try {
			String decryptData = SecurityUtil.decrypt(res.getEncryptData(), res.getEncryptType(), appSecret, appId);
			WebHook webhook = JSON.parseObject(decryptData, WebHook.class);

			// 参数转换
			responseParams.setAppId(appId);
			responseParams.setSign(res.getSign());
			responseParams.setSignType(res.getSignType());
			responseParams.setEncryptType(res.getEncryptType());
			responseParams.setTimestamp(res.getTimestamp());
			responseParams.setRespCode(res.getRespCode());
			responseParams.setRespMsg(res.getRespMsg());
			responseParams.setPageParams(res.getPageParams());
			responseParams.setTransType(res.getTransType());

			// 解密后设置param
			responseParams.setParam(webhook);
		} catch (Exception e1) {
			responseParams.setRespCode("-1");
			responseParams.setRespMsg("解密失败");
		}
		return responseParams;
	}

	/**
	 * 解密并验签响应消息
	 * 
	 * @Description:
	 * @Author: chenjh @Date: 2016年12月21日
	 * @LastEditTime:
	 * @param responseParams
	 * 			respCode : -1 返回解密失败
	 * 					   -2 返回签名失败
	 * @param appId
	 * @param appSecret
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<?> decryptVerifyWebHookResponse(ResponseParams<?> responseParams) throws PayException {
		ResponseParams<?> res = decryptWebHookResponse(responseParams);

		if (!mainService.verifySign(res)) {
			res.setRespCode("-2");
			res.setRespMsg("返回签名校验失败");
		}

		return res;
	}
//
//	/**
//	 * 请求报文验签
//	 * 
//	 * @Description:
//	 * @Author: chenjh @Date: 2016年10月20日
//	 * @LastEditTime:
//	 * @param requestMessage
//	 *            请求报文
//	 * @return
//	 * @throws PayException
//	 */
//	public boolean verifyRequestSign(String requestMessage) throws PayException {
//		return mainService.verifyRequestSign(requestMessage);
//	}

	/**
	 * 验签页面跳转参数
	 * 
	 * @param chargeAmt
	 *            订单金额
	 * @param outChargeNo
	 *            商户订单号
	 * @param status
	 *            订单状态
	 * @param channel
	 *            订单渠道
	 * @param timestamp
	 *            时间戳
	 * @param sign
	 *            签名
	 * @return
	 * @throws PayException
	 */
	public boolean verifyReturnSign(int chargeAmt, String outChargeNo, String status, String channel, String timestamp, String sign) throws PayException {
		return mainService.verifySign(chargeAmt, outChargeNo, status, channel, timestamp, sign);
	}

	/**
	 * 通过回调请求参数获取webhook对象
	 * 
	 * @Description:
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime:
	 * @param requestMessage
	 * @return
	 * @throws PayException
	 */
	public WebHook getWebHook(String requestMessage) throws PayException {
		return mainService.getWebHook(requestMessage);
	}

	/**
	 * 获取收银台请求参数
	 * 
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param params
	 * @return
	 * @throws PayException
	 */
	public String getCashierParams(ChargeParams params) throws PayException {
		return mainService.getCashierParams(params);
	}

	/**
	 * 创建收银台连接
	 * 
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param params
	 *            创建收银台入参
	 * @param encode
	 *            是否用UrlEncode编码转换
	 * @return 返回收银台连接地址
	 * @throws PayException
	 */
	public String createCashier(ChargeParams params, boolean encode) throws PayException {
		return mainService.createCashier(params, encode);
	}

	/**
	 * 创建收银台订单
	 * 
	 * @param params
	 *            创建收银台入参
	 * @param encode
	 *            是否用UrlEncode编码转换
	 * @Author: chenjh @Date: 2016年10月20日
	 * @return 返回收银台连接地址
	 * @throws PayException
	 */
	public ResponseParams<ChargeResult> initCashier(ChargeParams params) throws PayException {
		return mainService.initCashier(params);
	}

	/**
	 * 查询开通支付渠道列表
	 * 
	 * @Author: chenjh @Date: 2016年10月20日
	 * @param string
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<List<ChannelVO>> channelList(String platType) throws PayException {
		return mainService.channelList(platType);
	}

	/**
	 * 判断返回是否成功
	 * @return
	 */
	public static boolean isSuccessful(ResponseParams<?> res) {
		if (res != null && "000000".equals(res.getRespCode())) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 单独通过链接获取到openId密文之后需通过此接口来解密获取明文
	 * @author xuwy 2017/2/15
	 */
	public static String decryptWxOpenId (String encryptOpenId) throws PayException {
		final String SECURITY_LOCAL_KEY = "yht#2017";
		final String SECURITY_LOCAL_SECRET = "123456";
		if (StringUtil.isEmpty(encryptOpenId)) {
			throw new PayException("微信openId密文为空,请先通过授权地址获取openId密文");
		}
		try {
			return SecurityUtil.decrypt(encryptOpenId, EncType.AES.name(), SECURITY_LOCAL_SECRET, SECURITY_LOCAL_KEY);
		} catch (Exception e) {
			throw new PayException("微信openId解密失败,请重新授权获取");
		}
	}
	
	/**
	 * 下载对账文件
	 * @Author: chenshubi @Date: 2017年02月6日
	 * @param downloadBillParams 下载对账参数
	 * @return 退款返回
	 * @throws Exception
	 */
	public ResponseParams<DownloadBillResult> downloadBill(DownloadBillParams downloadBillParams) throws Exception {
		return mainService.downloadBill(downloadBillParams);
	}

	/**
	 * 获取商户应用二维码
	 * @param qrCodeParams
	 * @return
	 * @throws Exception
     */
	public ResponseParams<QRCodeResult> queryQRCode (QRCodeParams qrCodeParams) throws Exception {
		return mainService.queryQRCode(qrCodeParams);
	}
	
	/**
	 * 银企直连转账（妇幼）
	 * @param transferParams
	 * @return
	 * @throws PayException 
	 */
	public ResponseParams<TransferResult> transfer(TransferParams transferParams) throws PayException{
		return mainService.transfer(transferParams);
	}
	
	public ResponseParams<Object> passiveSiCardInfo(SocialCardParams socialCardParams) throws PayException{
		return mainService.passiveSiCardInfo(socialCardParams);
	}
	public ResponseParams<Object> siCardInfo(SocialCardParams socialCardParams) throws PayException{
		return mainService.siCardInfo(socialCardParams);
	}
	/**
	 * 生成社保请求医保支付地址
	 * @description 加密方默认传AES
	 * @author YANGD
	 * @date 2017年6月20日 下午3:06:48 
	 * @param param 必填 : accountId 微信公众号ID openId 微信openid 
	 * @return 医保请求支付地址 payUrl
	 * @throws PayException
	 */
	public ResponseParams<JSONObject> createSicardPayUrl(JSONObject socialCardParams) throws PayException{
		return mainService.createSicardPayUrl(socialCardParams);
	}
	
	/**
	 *   通过住院号，获取签约信息
	 * @param signInfoParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<SignInfoResult> querySignInfoByUserId(SignInfoParams signInfoParams) throws Exception {
		return mainService.signInfoByUserId(signInfoParams);
	}
	
	
	/**
	 * 操作人员——获取最新的签约信息
	 * @param signInfoParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<SignInfoResult> signInfo(SignInfoParams signInfoParams) throws Exception {
		return mainService.signInfo(signInfoParams);
	}
	
	/**
	 * pos 提交补充签约信息
	 * @param posSignInfoParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<PosSignInfoResult> posSignInfo(PosSignInfoParams posSignInfoParams) throws Exception {
		return mainService.posSignInfo(posSignInfoParams);
	}
	
	
	/**
	 *  pos 直接提交 全部的签约信息
	 * @param signInfoParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<SignInfoResult> saveSignInfo(SignInfoParams signInfoParams) throws Exception {
		return mainService.saveSignInfo(signInfoParams);
	}
	
	/**
	 * pos 腕带退款
	 * @param posSignInfoParams
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<WristbandRefundResult> wristbandRefund(WristbandRefundParams wristbandRefundParams) throws Exception {
		return mainService.wristbandRefund(wristbandRefundParams);
	}
	
	
	
	
	
	
	
	
	
}
