package com.ylzinfo.onepay.sdk.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ylzinfo.onepay.sdk.OnepayClient;
import com.ylzinfo.onepay.sdk.domain.ChannelVO;
import com.ylzinfo.onepay.sdk.domain.RequestParams;
import com.ylzinfo.onepay.sdk.domain.ResponseParams;
import com.ylzinfo.onepay.sdk.domain.WebHook;
import com.ylzinfo.onepay.sdk.enums.TransType;
import com.ylzinfo.onepay.sdk.enums.URLType;
import com.ylzinfo.onepay.sdk.exception.PayException;
import com.ylzinfo.onepay.sdk.utils.DateUtil;
import com.ylzinfo.onepay.sdk.utils.HttpUtil;
import com.ylzinfo.onepay.sdk.utils.SecurityUtil;
import com.ylzinfo.onepay.sdk.utils.Signature;
import com.ylzinfo.onepay.sdk.utils.StringUtil;

public class MainService {
	/**
	 * 
	 */
	protected static final String API_URI = "/oneweb/oneapi";
	protected static final String CASHIER_URI = "/oneweb/cashier";
	protected static final String MMPAY_URI = "/h5/hosPay";
	/**
	 * 服务地址
	 */
	protected String onepayUrl = "http://127.0.0.1:8080/onepay-web";

	/**
	 * 接口服务地址
	 */
	protected String apiUrl = onepayUrl + API_URI;

	/**
	 * 收银台服务地址
	 */
	protected String payPlatUrl = onepayUrl + CASHIER_URI;
	/**
	 * 医保结算地址
	 */
	protected String mmpayUrl = onepayUrl + MMPAY_URI;

	/**
	 * 应用id
	 */
	protected String appId;

	/**
	 * 应用安全key
	 */
	protected String appSecret;
	
	/**
	 * 签名类型
	 */
	protected String signType;
	
	/**
	 * 加密类型
	 */
	protected String encryptType;

	/**
	 * 统一支付平台SDK入口 
	 * @param onepayUrl 支付平台服务地址
	 * @param appId 应用编号
	 * @param appSecret 应用密钥
	 */
	public MainService(String onepayUrl, String appId, String appSecret, String signType, String encryptType) {
		super();
		this.onepayUrl = onepayUrl;
		apiUrl = onepayUrl + API_URI;
		payPlatUrl = onepayUrl + CASHIER_URI;
		mmpayUrl = onepayUrl + MMPAY_URI;
		
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
	}

	/**
	 * 统一支付平台SDK入口 ，默认平台地址
	 * @param appId 应用编号
	 * @param appSecret 应用密钥
	 */
	public MainService(String appId, String appSecret, String signType, String encryptType) {
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
	}

	/**
	 * 通用业务处理方法
	 * 
	 * @param transType
	 * @param params
	 * @return
	 * @throws PayException
	 */
	public <T> ResponseParams<T> doBusiness(String transType, Object params, Class<T> clz) throws PayException {
		try {
			String timestamp = DateUtil.getCurrentDateTime();

			// 创建请求参数对象
			RequestParams requestParams = new RequestParams();
			requestParams.setAppId(appId);
			requestParams.setTimestamp(timestamp);
			requestParams.setParam(params);
			requestParams.setTransType(transType);
			requestParams.setSignType(signType);
			requestParams.setEncryptType(encryptType);

			// 创建签名信息
			String sign = Signature.createSign(requestParams, appSecret);
			requestParams.setSign(sign);
			
			// 加密报文
			try {
				System.out.println("加密前报文：" + JSONObject.toJSONString(requestParams));
				String encryptData = SecurityUtil.encrypt(JSONObject.toJSONString(requestParams.getParam()), encryptType, appSecret, appId);
				System.out.println("加密后报文：" + encryptData);
				requestParams.setEncryptData(encryptData);
			} catch (Exception e) {
				throw new PayException("请求报文加密失败");
			}
			
			// 清空明文
			requestParams.setParam(null);

			// 创建请求报文并发送请求
			String requestMessage = JSON.toJSONString(requestParams);
			System.out.println("请求参数报文：" + requestMessage);
			URLType urlType = URLType.HTTP;
			
			if (apiUrl.toLowerCase().startsWith("https")) {
				urlType = URLType.HTTPS;
			}
			
			String responseMsg = HttpUtil.request(apiUrl, requestMessage, urlType);

			if (StringUtil.isEmpty(responseMsg)) {
				throw new PayException("请求错误");
			}
		
			// 参数转换
			JSONObject res = JSON.parseObject(responseMsg);

			ResponseParams<T> responseParams = new ResponseParams<T>();
			responseParams.setSign(res.getString("sign"));
			responseParams.setSignType(res.getString("signType"));
			responseParams.setEncryptType(res.getString("encryptType"));
			responseParams.setTimestamp(res.getString("timestamp"));
			responseParams.setRespCode(res.getString("respCode"));
			responseParams.setRespMsg(res.getString("respMsg"));
			responseParams.setBillContent(res.getString("billContent"));
			
			// 解密报文
			try {
				String encData = res.getString("encryptData");
				
				if (StringUtil.isNotEmpty(encData)) {
					System.out.println("解密前报文：" + JSONObject.toJSONString(responseParams));
					String decryptData = SecurityUtil.decrypt(encData, encryptType, appSecret, appId);
					res.put("param", JSON.parseObject(decryptData));
					responseParams.setParam(JSON.parseObject(decryptData, clz));
					System.out.println("解密后报文：" + JSONObject.toJSONString(responseParams));
				}
			} catch (Exception e) {
				throw new PayException("响应报文解密失败");
			}

			// 校验返回签名
			if (OnepayClient.isSuccessful(responseParams) && !verifySign(res)) {
				throw new PayException("验签失败");
			}
			
			return responseParams;
		} catch (Exception e) {
			e.printStackTrace();
			throw new PayException(e);
		}
	}

	/**
	 * 通用处理方法，适合返回列表请求
	 * @param transType
	 * @param params
	 * @param type
	 * @return
	 * @throws PayException
	 */
	public <T> ResponseParams<List<T>> doBusinessArray(String transType, Object params, Class<T> clz) throws PayException {

		try {
			String timestamp = DateUtil.getCurrentDateTime();
			// 创建请求参数对象
			RequestParams requestParams = new RequestParams();
			requestParams.setAppId(appId);
			requestParams.setTimestamp(timestamp);
			requestParams.setParam(params);
			requestParams.setTransType(transType);
			requestParams.setSignType(signType);
			requestParams.setEncryptType(encryptType);

			// 创建签名信息
			String sign = Signature.createSign(requestParams, appSecret);
			requestParams.setSign(sign);
			
			// 加密报文
			System.out.println("加密前报文：" + JSONObject.toJSONString(requestParams.getParam()));
			String encryptData = SecurityUtil.encrypt(JSONObject.toJSONString(requestParams.getParam()), encryptType, appSecret, appId);
			System.out.println("加密后报文：" + encryptData);
			requestParams.setEncryptData(encryptData);
			
			// 清空明文
			requestParams.setParam(null);

			// 创建请求报文并发送请求
			String requestMessage = JSON.toJSONString(requestParams);
			System.out.println("请求参数报文：" + requestMessage);
			String responseMsg = HttpUtil.request(apiUrl, requestMessage, URLType.HTTP);

			if (StringUtil.isEmpty(responseMsg)) {
				throw new PayException("请求错误");
			}
					
			// 参数转换
			JSONObject res = JSON.parseObject(responseMsg);
			
			// 参数转换
			ResponseParams<List<T>> responseParams = new ResponseParams<List<T>>();
			responseParams.setSign(res.getString("sign"));
			responseParams.setSignType(res.getString("signType"));
			responseParams.setEncryptType(res.getString("encryptType"));
			responseParams.setTimestamp(res.getString("timestamp"));
			responseParams.setRespCode(res.getString("respCode"));
			responseParams.setRespMsg(res.getString("respMsg"));
			
			try {
				// 解密报文
				String encData = res.getString("encryptData");
				System.out.println("解密前报文：" + encData);
				String decryptData = SecurityUtil.decrypt(encData, encryptType, appSecret, appId);
				System.out.println("解密后报文：" + decryptData);
				
				// 解密结果
				res.put("param", JSONArray.parseArray(decryptData));
				responseParams.setParam(JSONArray.parseArray(decryptData, clz));
			} catch (Exception e) {
				throw new PayException("响应报文解密失败");
			}

			// 校验返回签名
			if (OnepayClient.isSuccessful(responseParams) && !verifySign(res)) {
				throw new PayException("验签失败");
			}

			return responseParams;
		} catch (Exception e) {
			e.printStackTrace();
			throw new PayException(e);
		}
	}
	
	public static void main(String[] args) {
		String res = "[{\"sortNo\":\"0\",\"channelNote\":\"请使用银联卡支付，无需开通网银\",\"appId\":\"1A3VL0KVK0000B020A0A0000CC3F48AD\",\"channelImg\":\"/static/img/platimg/appimg/pay-icon/wx-pay.png\",\"channelName\":\"微信支付\",\"subChannelId\":\"WX_QR\",\"channelId\":\"WX\"},{\"sortNo\":\"1\",\"channelNote\":\"使用微信支付，安全快捷\",\"appId\":\"1A3VL0KVK0000B020A0A0000CC3F48AD\",\"channelImg\":\"/static/img/platimg/appimg/pay-icon/ali-pay.png\",\"channelName\":\"支付宝\",\"subChannelId\":\"ALI_WEB\",\"channelId\":\"ALI\"},{\"sortNo\":\"2\",\"channelNote\":\"推荐有支付宝账号的用户使用\",\"appId\":\"1A3VL0KVK0000B020A0A0000CC3F48AD\",\"channelImg\":\"/static/img/platimg/appimg/pay-icon/union-pay.png\",\"channelName\":\"银联支付\",\"subChannelId\":\"UP_WEB\",\"channelId\":\"UP\"}]";
		List<ChannelVO>  list = JSONArray.parseArray(res, ChannelVO.class);
		ChannelVO co = list.get(0);
		System.out.println(co);
	}

	/**
	 * 验证签名
	 * @param res
	 * @return
	 * @throws PayException 
	 */
	private boolean verifySign(JSONObject res) throws PayException {
		try {
			String oldSign = res.getString("sign");
			String newSign = Signature.createSign(res, appSecret);
			return oldSign.equals(newSign);
		} catch (Exception e) {
			throw new PayException("验证失败, " + e.getMessage());
		}
	}
	
	/**
	 * 验证签名
	 * 
	 * @param responseMsg
	 * @return
	 * @throws PayException
	 */
	public boolean verifySign(String responseMsg) throws PayException {
		return verifySign(JSON.parseObject(responseMsg));
	}

	/**
	 * 验证签名
	 * 
	 * @param responseMsg
	 * @return
	 * @throws PayException
	 */
	public boolean verifySign(ResponseParams<?> responseParams) throws PayException {
		return verifySign(JSON.parseObject(JSON.toJSONString(responseParams)));
	}

	/**
	 * 验证请求签名
	 * @param requestMessage
	 * @return
	 * @throws PayException
	 */
	public boolean verifyRequestSign(String requestMessage) throws PayException {
		return verifySign(JSON.parseObject(requestMessage));
	}
	
	/**
	 * 回调
	 * @param requestMessage
	 * @return
	 * @throws PayException
	 */
	public WebHook getWebHook(String requestMessage) throws PayException {
		try {
			RequestParams requestParams = JSON.parseObject(requestMessage, RequestParams.class);
			WebHook webHook = JSON.parseObject(JSON.toJSONString(requestParams.getParam()), WebHook.class);
			return webHook;
		} catch (Exception e) {
			throw new PayException("回调失败, " + e);
		}
	}

	/**
	 * 验证页面跳转参数签名
	 * @param chargeAmt
	 * @param outChargeNo
	 * @param status
	 * @param channel
	 * @param timestamp
	 * @param sign
	 * @return
	 * @throws PayException 
	 */
	public boolean verifySign(int chargeAmt, String outChargeNo, String status, String channel, String timestamp, String sign) throws PayException {
		try {
			String oldSign = sign;
			String newSign = Signature.createSign(appId, chargeAmt, outChargeNo, status, channel, timestamp, appSecret, signType);
			return oldSign.equals(newSign);
		} catch (Exception e) {
			throw new PayException("验证失败, " + e.getMessage());
		}
	}

	/**
	 * 查询开通的支付渠道列表
	 * @param type
	 * @return
	 * @throws PayException 
	 */
	public ResponseParams<List<ChannelVO>> channelList(String platType) throws PayException {
		String transType = TransType.QUERY_CHANNEL_LIST.getCode();
		Map<String, String> params = new HashMap<String, String>();
		params.put("appId", appId);
		params.put("platType", platType);
		return doBusinessArray(transType, params, ChannelVO.class);
	}

	/**
	 * 通用业务处理方法
	 * 
	 * @param transType
	 * @param params
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<JSONObject> execute(RequestParams requestParams) throws PayException {
		try {
			String timestamp = DateUtil.getCurrentDateTime();
			// 创建请求参数对象
			if (StringUtil.isEmpty(requestParams.getAppId()))
				requestParams.setAppId(appId);
			if (StringUtil.isEmpty(requestParams.getTimestamp()))
				requestParams.setTimestamp(timestamp);
			if (StringUtil.isEmpty(requestParams.getSignType()))
				requestParams.setSignType(signType);
			if (StringUtil.isEmpty(requestParams.getEncryptType()))
				requestParams.setEncryptType(encryptType);

			// 创建签名信息
			String sign = Signature.createSign(requestParams, appSecret);
			requestParams.setSign(sign);
			
			// 加密报文
			try {
				System.out.println("加密前报文：" + JSONObject.toJSONString(requestParams));
				String encryptData = SecurityUtil.encrypt(JSONObject.toJSONString(requestParams.getParam()), encryptType, appSecret, appId);
				System.out.println("加密后报文：" + encryptData);
				requestParams.setEncryptData(encryptData);
			} catch (Exception e) {
				throw new PayException("请求报文加密失败");
			}
			
			// 清空明文
			requestParams.setParam(null);

			// 创建请求报文并发送请求
			String requestMessage = JSON.toJSONString(requestParams);
			System.out.println("请求参数报文：" + requestMessage);
			URLType urlType = URLType.HTTP;
			
			if (apiUrl.toLowerCase().startsWith("https")) {
				urlType = URLType.HTTPS;
			}
			
			String responseMsg = HttpUtil.request(apiUrl, requestMessage, urlType);

			if (StringUtil.isEmpty(responseMsg)) {
				throw new PayException("请求错误");
			}
		
			// 参数转换
			JSONObject res = JSON.parseObject(responseMsg);

			ResponseParams<JSONObject> responseParams = new ResponseParams<JSONObject>();
			responseParams.setSign(res.getString("sign"));
			responseParams.setSignType(res.getString("signType"));
			responseParams.setEncryptType(res.getString("encryptType"));
			responseParams.setTimestamp(res.getString("timestamp"));
			responseParams.setRespCode(res.getString("respCode"));
			responseParams.setRespMsg(res.getString("respMsg"));
			responseParams.setBillContent(res.getString("billContent"));
			// 解密报文
			try {
				String encData = res.getString("encryptData");
				
				if (StringUtil.isNotEmpty(encData)) {
					System.out.println("解密前报文：" + JSONObject.toJSONString(responseParams));
					String decryptData = SecurityUtil.decrypt(encData, encryptType, appSecret, appId);
					res.put("param", JSON.parseObject(decryptData));
					responseParams.setParam(JSON.parseObject(decryptData));
					System.out.println("解密后报文：" + JSONObject.toJSONString(responseParams));
				}
			} catch (Exception e) {
				throw new PayException("响应报文解密失败");
			}

			// 校验返回签名
			if (OnepayClient.isSuccessful(responseParams) && !verifySign(res)) {
				throw new PayException("验签失败");
			}
			
			return responseParams;
		} catch (Exception e) {
			e.printStackTrace();
			throw new PayException(e);
		}
	}

	public String getOnepayUrl() {
		return onepayUrl;
	}

	public void setOnepayUrl(String onepayUrl) {
		this.onepayUrl = onepayUrl;
	}

	public String getApiUrl() {
		return apiUrl;
	}

	public void setApiUrl(String apiUrl) {
		this.apiUrl = apiUrl;
	}

	public String getPayPlatUrl() {
		return payPlatUrl;
	}

	public void setPayPlatUrl(String payPlatUrl) {
		this.payPlatUrl = payPlatUrl;
	}

	public String getAppId() {
		return appId;
	}

	public void setAppId(String appId) {
		this.appId = appId;
	}

	public String getAppSecret() {
		return appSecret;
	}

	public void setAppSecret(String appSecret) {
		this.appSecret = appSecret;
	}

}
