package com.ylzinfo.onepay.sdk.service;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ylzinfo.onepay.sdk.domain.CancelParams;
import com.ylzinfo.onepay.sdk.domain.CancelResult;
import com.ylzinfo.onepay.sdk.domain.ChargeParams;
import com.ylzinfo.onepay.sdk.domain.ChargeResult;
import com.ylzinfo.onepay.sdk.domain.ChargeVO;
import com.ylzinfo.onepay.sdk.domain.CloseParams;
import com.ylzinfo.onepay.sdk.domain.CloseResult;
import com.ylzinfo.onepay.sdk.domain.DownloadBillParams;
import com.ylzinfo.onepay.sdk.domain.DownloadBillResult;
import com.ylzinfo.onepay.sdk.domain.PosSignInfoParams;
import com.ylzinfo.onepay.sdk.domain.PosSignInfoResult;
import com.ylzinfo.onepay.sdk.domain.QRCodeParams;
import com.ylzinfo.onepay.sdk.domain.QRCodeResult;
import com.ylzinfo.onepay.sdk.domain.QueryParams;
import com.ylzinfo.onepay.sdk.domain.RefundParams;
import com.ylzinfo.onepay.sdk.domain.RefundResult;
import com.ylzinfo.onepay.sdk.domain.RefundVO;
import com.ylzinfo.onepay.sdk.domain.RequestParams;
import com.ylzinfo.onepay.sdk.domain.ResponseParams;
import com.ylzinfo.onepay.sdk.domain.SignInfoParams;
import com.ylzinfo.onepay.sdk.domain.SignInfoResult;
import com.ylzinfo.onepay.sdk.domain.SocialCardParams;
import com.ylzinfo.onepay.sdk.domain.TransferParams;
import com.ylzinfo.onepay.sdk.domain.TransferResult;
import com.ylzinfo.onepay.sdk.domain.WristbandRefundParams;
import com.ylzinfo.onepay.sdk.domain.WristbandRefundResult;
import com.ylzinfo.onepay.sdk.enums.TransType;
import com.ylzinfo.onepay.sdk.exception.PayException;
import com.ylzinfo.onepay.sdk.utils.DateUtil;
import com.ylzinfo.onepay.sdk.utils.SecurityUtil;
import com.ylzinfo.onepay.sdk.utils.Signature;

public class OnepayService extends MainService{
	
	/**
	 * 统一支付平台SDK入口 
	 * @param onepayUrl 支付平台服务地址
	 * @param appId 应用编号
	 * @param appSecret 应用密钥
	 * @param singType 签名类型:MD5、RSA
	 * @param encryptType 加密类型
	 */
	public OnepayService(String onepayUrl, String appId, String appSecret, String signType, String encryptType) {
		super(onepayUrl, appId, appSecret, signType, encryptType);
	}
	/**
	 * 统一支付平台SDK入口 
	 * @param appId 应用编号
	 * @param appSecret 应用密钥
	 * @param singType 签名类型:MD5、RSA
	 * @param encryptType 加密类型
	 */
	public OnepayService(String appId, String appSecret, String signType, String encryptType) {
		super(appId, appSecret, signType, encryptType);
	}

	/**
	 * 支付
	 * 
	 * @param chargeParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<ChargeResult> charge(ChargeParams chargeParams) throws PayException {
		String transType = TransType.CHARGE.getCode();
		return doBusiness(transType, chargeParams, ChargeResult.class);
	}

	/**
	 * 退款
	 * 
	 * @param refundParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<RefundResult> refund(RefundParams refundParams) throws PayException {
		String transType = TransType.REFUND.getCode();
		return doBusiness(transType, refundParams, RefundResult.class);
	}
	
	/**
	 * 下载对账文件
	 * 
	 * @param refundParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<DownloadBillResult> downloadBill(DownloadBillParams downloadBillParams) throws PayException {
		String transType = TransType.DOWNLOAD_BILL.getCode();
		return doBusiness(transType, downloadBillParams, DownloadBillResult.class);
	}
	
	/**
	 * 关闭
	 * 
	 * @param closeParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<CloseResult> close(CloseParams closeParams) throws PayException {
		String transType = TransType.CLOSE.getCode();
		return doBusiness(transType, closeParams, CloseResult.class);
	}

	/**
	 * 撤单
	 */
	public ResponseParams<CancelResult> cancel(CancelParams cancelParams) throws PayException {
		String transType = TransType.CANCEL.getCode();
		return doBusiness(transType, cancelParams, CancelResult.class);
	}

	/**
	 * 查询支付列表
	 * 
	 * @param queryParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<List<ChargeVO>> queryChargeList(QueryParams queryParams) throws PayException {
		String transType = TransType.QUERY_CHARGE_LIST.getCode();
		return doBusinessArray(transType, queryParams, ChargeVO.class);
	}

	/**
	 * 查询退款列表
	 * 
	 * @param queryParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<List<RefundVO>> queryRefundList(QueryParams queryParams) throws PayException {
		String transType = TransType.QUERY_REFUND_LIST.getCode();
		return doBusinessArray(transType, queryParams, RefundVO.class);
	}

	/**
	 * 查询支付订单
	 * 
	 * @param chargeNo
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<ChargeVO> queryCharge(String chargeNo) throws PayException {
		String transType = TransType.QUERY_CHARGE.getCode();
		Map<String, String> params = new HashMap<String, String>();
		params.put("outChargeNo", chargeNo);
		return doBusiness(transType, params, ChargeVO.class);
	}

	/**
	 * 查询退款订单
	 * 
	 * @param refundNo
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<RefundVO> queryRefund(String refundNo) throws PayException {
		String transType = TransType.QUERY_REFUND.getCode();
		Map<String, String> params = new HashMap<String, String>();
		params.put("outRefundNo", refundNo);
		return doBusiness(transType, params, RefundVO.class);
	}

	/**
	 * 组装请求参数
	 * 
	 * @param params
	 * @return
	 * @throws PayException
	 */
	public String getCashierParams(ChargeParams params) throws PayException {
		try {
			RequestParams requestParams = new RequestParams();
			String timestamp = DateUtil.getCurrentDateTime();
			requestParams.setAppId(appId);
			requestParams.setSignType(signType);
			requestParams.setEncryptType(encryptType);
			requestParams.setTimestamp(timestamp);
			requestParams.setParam(params);
			requestParams.setTransType(TransType.CASHIER_CHARGE.getCode());
			
			// 创建签名信息
			String sign = Signature.createSign(requestParams, appSecret);
			
			requestParams.setSign(sign);
			// 加密报文
			System.out.println("加密前报文：" + JSONObject.toJSONString(requestParams.getParam()));
			String encryptData = SecurityUtil.encrypt(JSONObject.toJSONString(requestParams.getParam()), encryptType, appSecret, appId);
			System.out.println("加密后报文：" + encryptData);
			requestParams.setEncryptData(encryptData);
			// 清空明文
			requestParams.setParam(null);
			
			// 创建请求报文并发送请求
			String requestMessage = JSON.toJSONString(requestParams);
			return requestMessage;
		} catch (Exception e) {
			throw new PayException(e);
		}
	}

	/**
	 * 收银地址
	 * 
	 * @param params
	 * @return
	 * @throws PayException
	 */
	public String createCashier(ChargeParams params, boolean encode) throws PayException {
		try {
			return payPlatUrl + "?param=" + (encode ? URLEncoder.encode(getCashierParams(params), "utf-8") : getCashierParams(params));
		} catch (UnsupportedEncodingException e) {
			throw new PayException("收银地址生成失败");
		}
	}

	/**
	 * 初始化收银台
	 * @param params
	 * @param encode
	 * @return
	 * @throws PayException 
	 */
	public ResponseParams<ChargeResult> initCashier(ChargeParams chargeParams) throws PayException {
		String transType = TransType.CASHIER_PREORDER.getCode();
		return doBusiness(transType, chargeParams, ChargeResult.class);
	}

	/**
	 * 获取商户二维码
	 */
	public ResponseParams<QRCodeResult> queryQRCode(QRCodeParams qrCodeParams) throws PayException {
		String transType = TransType.QUERY_QRCODE.getCode();
		return doBusiness(transType, qrCodeParams, QRCodeResult.class);
	}
	
	/**
	 * 银企直连 转账
	 * @author hshib
	 * @date 2017/05/04
	 * @param transferParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<TransferResult> transfer(TransferParams transferParams) throws PayException{
		String transType=TransType.OP_TRADE_TRANSFER.getCode();
		return doBusiness(transType, transferParams, TransferResult.class);
	}
	
	/**
	 * 医院医保绑卡信息查询
	 * @author hshib
	 * @date 2017/06/13
	 * @param socialCardParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<Object> siCardInfo(SocialCardParams socialCardParams) throws PayException{
		String transType=TransType.HOS_SOCIAL_CARD_QUERY.getCode();
		return doBusiness(transType, socialCardParams, Object.class);
	}
	/** 
	* @Title: passiveSiCardInfo 
	* @Description: TODO(查询被办卡信息) 
	* @param socialCardParams
	* @return
	* @throws PayException  参数说明 
	* ResponseParams<Object>    返回类型 
	* @throws 
	*/
	public ResponseParams<Object> passiveSiCardInfo(SocialCardParams socialCardParams) throws PayException{
		String transType=TransType.HOS_SOCIAL_BIND_QUERY.getCode();
		return doBusiness(transType, socialCardParams, Object.class);
	}
	/**
	 * 生成社保请求医保支付地址
	 * @author YANGD
	 * @date 2017年6月20日 下午3:01:14 
	 * @param param accountId 微信公众号ID openId 微信openid
	 * @return 医保支付的url地址
	 * @throws PayException
	 */
	public ResponseParams<JSONObject> createSicardPayUrl(JSONObject plainJson) throws PayException{
		ResponseParams<JSONObject> responseParams = new ResponseParams<JSONObject>();
		String payUrl = "";
		try {
			String aesdata = SecurityUtil.encrypt(JSONObject.toJSONString(plainJson), encryptType, appSecret, appId);
			payUrl = mmpayUrl + "?appId=" + appId + "&aesdata=" + aesdata;;
		} catch (Exception e) {
			throw new PayException(e);
		}
		JSONObject resParam = new JSONObject();
		resParam.put("payUrl", payUrl);
		responseParams.setRespCode("000000");
		responseParams.setParam(resParam);
		return responseParams;
	}
	
	/**
	 * 操作员——签约查询  最新一条
	 * @param signInfoParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<SignInfoResult> signInfo(SignInfoParams signInfoParams) throws PayException {
		String transType = TransType.SIGN_INFO.getCode();
		return doBusiness(transType, signInfoParams, SignInfoResult.class);
	}
	
	/**
	 * 住院号(userId)——签约查询  唯一一条
	 * @param signInfoParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<SignInfoResult> signInfoByUserId(SignInfoParams signInfoParams) throws PayException {
		String transType = TransType.SIGN_INFO_USERID.getCode();
		return doBusiness(transType, signInfoParams, SignInfoResult.class);
	}
	/**
	 * pos 补充签约信息
	 * @param posSignInfoParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<PosSignInfoResult> posSignInfo(PosSignInfoParams posSignInfoParams) throws PayException {
		String transType = TransType.POS_SIGN_INFO.getCode();
		return doBusiness(transType, posSignInfoParams, PosSignInfoResult.class);
	}
	
	/**
	 * pos签约 直接提交全部的签约信息
	 * @param posSignInfoParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<SignInfoResult> saveSignInfo(SignInfoParams signInfoParams) throws PayException {
		String transType = TransType.SAVE_SIGNINFO.getCode();
		return doBusiness(transType, signInfoParams, SignInfoResult.class);
	}
	
	/**
	 * 
	 * @param wristbandRefundParams
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<WristbandRefundResult> wristbandRefund(WristbandRefundParams wristbandRefundParams) throws PayException {
		String transType = TransType.WRISTBAND_REFUND.getCode();
		return doBusiness(transType, wristbandRefundParams, WristbandRefundResult.class);
	}
}
