package com.ylzinfo.onepay.sdk;

import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ylzinfo.onepay.sdk.domain.RequestParams;
import com.ylzinfo.onepay.sdk.domain.ResponseParams;
import com.ylzinfo.onepay.sdk.enums.EncType;
import com.ylzinfo.onepay.sdk.enums.SignType;
import com.ylzinfo.onepay.sdk.exception.PayException;
import com.ylzinfo.onepay.sdk.service.OnepayService;
import com.ylzinfo.onepay.sdk.utils.SecurityUtil;
import com.ylzinfo.onepay.sdk.utils.Signature;

/**
 * 支付平台客户端
 * 
 * @author chenjh
 * 
 */
public class OnepayDefaultClient {
	/**
	 * 应用id
	 */
	private String appId;

	/**
	 * 应用安全key
	 */
	private String appSecret;

	/**
	 * 签名类型
	 */
	private String signType;

	/**
	 * 加密类型
	 */
	private String encryptType;

	/**
	 * 应用服务
	 */
	private OnepayService mainService;

	public OnepayDefaultClient(String onepayUrl, String appId, String appSecret) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = SignType.MD5.toString();
		this.encryptType = EncType.DES.toString();
		mainService = new OnepayService(onepayUrl, appId, appSecret, signType, encryptType);
	}

	public OnepayDefaultClient(String onepayUrl, String appId, String appSecret, String signType, String encryptType) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
		mainService = new OnepayService(onepayUrl, appId, appSecret, signType, encryptType);
	}

	public OnepayDefaultClient(String appId, String appSecret, String signType, String encryptType) {
		super();
		this.appId = appId;
		this.appSecret = appSecret;
		this.signType = signType;
		this.encryptType = encryptType;
		mainService = new OnepayService(appId, appSecret, signType, encryptType);
	}

	public String getAppId() {
		return appId;
	}

	public String getAppSecret() {
		return appSecret;
	}

	public String getSignType() {
		return signType;
	}

	public String getEncryptType() {
		return encryptType;
	}

	/**
	 * 订单支付
	 * 
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime :
	 * @param payParams
	 *            支付请求参数
	 * @return
	 * @throws Exception
	 */
	public ResponseParams<JSONObject> execute(RequestParams requestParams) throws Exception {
		return mainService.execute(requestParams);
	}

	/**
	 * 解密异步通知响应消息
	 * 
	 * @Description:
	 * @Author: chenjh @Date: 2016年12月21日
	 * @LastEditTime:
	 * @param responseParams
	 * @param appId
	 * @param appSecret
	 * @return
	 * @throws PayException
	 */
	public ResponseParams<JSONObject> decryptNotifyResponse(ResponseParams<?> res) throws PayException {
		// 解密报文
		ResponseParams<JSONObject> responseParams = new ResponseParams<JSONObject>();

		try {
			String decryptData = SecurityUtil.decrypt(res.getEncryptData(), res.getEncryptType(), appSecret, appId);
			JSONObject webhook = JSON.parseObject(decryptData);

			// 参数转换
			responseParams.setAppId(appId);
			responseParams.setSign(res.getSign());
			responseParams.setSignType(res.getSignType());
			responseParams.setEncryptType(res.getEncryptType());
			responseParams.setTimestamp(res.getTimestamp());
			responseParams.setRespCode(res.getRespCode());
			responseParams.setRespMsg(res.getRespMsg());
			responseParams.setPageParams(res.getPageParams());
			responseParams.setTransType(res.getTransType());

			// 解密后设置param
			responseParams.setParam(webhook);
		} catch (Exception e1) {
			responseParams.setRespCode("-1");
			responseParams.setRespMsg("解密失败");
		}
		return responseParams;
	}

	/**
	 * returnUrl 验签
	 * 
	 * @Author: chenjh @Date: 2016年10月20日
	 * @LastEditTime:
	 * @param responseMsg
	 * @return
	 * @throws Exception
	 * @Description:
	 */
	public boolean verifyReturnSign(Map<String, String> returnParam, String sign) throws PayException {
		try {
			String oldSign = sign;
			String newSign = Signature.createSign(JSONObject.parseObject(JSON.toJSONString(returnParam)), appSecret);
			return oldSign.equals(newSign);
		} catch (Exception e) {
			throw new PayException("验证失败, " + e.getMessage());
		}
	}

	public boolean verifyResponseSign(ResponseParams<?> responseParams) throws PayException {
		return mainService.verifySign(responseParams);
	}
	
	/**
	 * 判断返回是否成功
	 * 
	 * @return
	 */
	public static boolean isSuccessful(ResponseParams<?> res) {
		if (res != null && "000000".equals(res.getRespCode())) {
			return true;
		} else {
			return false;
		}
	}
}
