/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.jvm;

import com.google.common.collect.ImmutableMap;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.jvm.JvmStats;
import org.elasticsearch.threadpool.ThreadPool;

public class JvmMonitorService
extends AbstractLifecycleComponent<JvmMonitorService> {
    private final ThreadPool threadPool;
    private final boolean enabled;
    private final TimeValue interval;
    private final ImmutableMap<String, GcThreshold> gcThresholds;
    private volatile ScheduledFuture scheduledFuture;

    @Inject
    public JvmMonitorService(Settings settings, ThreadPool threadPool) {
        super(settings);
        this.threadPool = threadPool;
        this.enabled = this.settings.getAsBoolean("monitor.jvm.enabled", (Boolean)true);
        this.interval = this.settings.getAsTime("monitor.jvm.interval", TimeValue.timeValueSeconds(1L));
        MapBuilder<String, GcThreshold> gcThresholds = MapBuilder.newMapBuilder();
        Map<String, Settings> gcThresholdGroups = this.settings.getGroups("monitor.jvm.gc");
        for (Map.Entry<String, Settings> entry : gcThresholdGroups.entrySet()) {
            String name = entry.getKey();
            TimeValue warn = entry.getValue().getAsTime("warn", null);
            TimeValue info = entry.getValue().getAsTime("info", null);
            TimeValue debug = entry.getValue().getAsTime("debug", null);
            if (warn == null || info == null || debug == null) {
                this.logger.warn("ignoring gc_threshold for [{}], missing warn/info/debug values", name);
                continue;
            }
            gcThresholds.put(name, new GcThreshold(name, warn.millis(), info.millis(), debug.millis()));
        }
        if (!gcThresholds.containsKey("young")) {
            gcThresholds.put("young", new GcThreshold("young", 1000L, 700L, 400L));
        }
        if (!gcThresholds.containsKey("old")) {
            gcThresholds.put("old", new GcThreshold("old", 10000L, 5000L, 2000L));
        }
        if (!gcThresholds.containsKey("default")) {
            gcThresholds.put("default", new GcThreshold("default", 10000L, 5000L, 2000L));
        }
        this.gcThresholds = gcThresholds.immutableMap();
        this.logger.debug("enabled [{}], interval [{}], gc_threshold [{}]", this.enabled, this.interval, this.gcThresholds);
    }

    @Override
    protected void doStart() {
        if (!this.enabled) {
            return;
        }
        this.scheduledFuture = this.threadPool.scheduleWithFixedDelay(new JvmMonitor(), this.interval);
    }

    @Override
    protected void doStop() {
        if (!this.enabled) {
            return;
        }
        FutureUtils.cancel(this.scheduledFuture);
    }

    @Override
    protected void doClose() {
    }

    private class JvmMonitor
    implements Runnable {
        private JvmStats lastJvmStats = JvmStats.jvmStats();
        private long seq = 0L;

        @Override
        public void run() {
            try {
                this.monitorLongGc();
            }
            catch (Throwable t) {
                JvmMonitorService.this.logger.debug("failed to monitor", t, new Object[0]);
            }
        }

        private synchronized void monitorLongGc() {
            ++this.seq;
            JvmStats currentJvmStats = JvmStats.jvmStats();
            for (int i = 0; i < currentJvmStats.getGc().getCollectors().length; ++i) {
                long avgCollectionTime;
                long collectionTime;
                JvmStats.GarbageCollector gc = currentJvmStats.getGc().getCollectors()[i];
                JvmStats.GarbageCollector prevGc = this.lastJvmStats.gc.collectors[i];
                long collections = gc.collectionCount - prevGc.collectionCount;
                if (collections == 0L || (collectionTime = gc.collectionTime - prevGc.collectionTime) == 0L) continue;
                GcThreshold gcThreshold = (GcThreshold)JvmMonitorService.this.gcThresholds.get((Object)gc.getName());
                if (gcThreshold == null) {
                    gcThreshold = (GcThreshold)JvmMonitorService.this.gcThresholds.get((Object)"default");
                }
                if ((avgCollectionTime = collectionTime / collections) > gcThreshold.warnThreshold) {
                    JvmMonitorService.this.logger.warn("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc.getName(), this.seq, gc.getCollectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.getTimestamp() - this.lastJvmStats.getTimestamp()), TimeValue.timeValueMillis(collectionTime), gc.getCollectionTime(), this.lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), JvmInfo.jvmInfo().getMem().getHeapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
                    continue;
                }
                if (avgCollectionTime > gcThreshold.infoThreshold) {
                    JvmMonitorService.this.logger.info("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc.getName(), this.seq, gc.getCollectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.getTimestamp() - this.lastJvmStats.getTimestamp()), TimeValue.timeValueMillis(collectionTime), gc.getCollectionTime(), this.lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), JvmInfo.jvmInfo().getMem().getHeapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
                    continue;
                }
                if (avgCollectionTime <= gcThreshold.debugThreshold || !JvmMonitorService.this.logger.isDebugEnabled()) continue;
                JvmMonitorService.this.logger.debug("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc.getName(), this.seq, gc.getCollectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.getTimestamp() - this.lastJvmStats.getTimestamp()), TimeValue.timeValueMillis(collectionTime), gc.getCollectionTime(), this.lastJvmStats.getMem().getHeapUsed(), currentJvmStats.getMem().getHeapUsed(), JvmInfo.jvmInfo().getMem().getHeapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
            }
            this.lastJvmStats = currentJvmStats;
        }

        private String buildPools(JvmStats prev, JvmStats current) {
            StringBuilder sb = new StringBuilder();
            for (JvmStats.MemoryPool currentPool : current.getMem()) {
                JvmStats.MemoryPool prevPool = null;
                for (JvmStats.MemoryPool pool : prev.getMem()) {
                    if (!pool.getName().equals(currentPool.getName())) continue;
                    prevPool = pool;
                    break;
                }
                sb.append("{[").append(currentPool.getName()).append("] [").append(prevPool == null ? "?" : prevPool.getUsed()).append("]->[").append(currentPool.getUsed()).append("]/[").append(currentPool.getMax()).append("]}");
            }
            return sb.toString();
        }
    }

    static class GcThreshold {
        public final String name;
        public final long warnThreshold;
        public final long infoThreshold;
        public final long debugThreshold;

        GcThreshold(String name, long warnThreshold, long infoThreshold, long debugThreshold) {
            this.name = name;
            this.warnThreshold = warnThreshold;
            this.infoThreshold = infoThreshold;
            this.debugThreshold = debugThreshold;
        }

        public String toString() {
            return "GcThreshold{name='" + this.name + '\'' + ", warnThreshold=" + this.warnThreshold + ", infoThreshold=" + this.infoThreshold + ", debugThreshold=" + this.debugThreshold + '}';
        }
    }
}

